#!/usr/bin/env bash
#
# wrap_pgp.sh – Add fake PGP delimiters around a Base‑64 string
#
# Usage:
#   ./wrap_pgp.sh <b64_input> <pgp_output>
#
#   <b64_input>   – file that contains a single line of Base‑64 data
#   <pgp_output>  – file that will receive the fake PGP block
#
# The script does **no real OpenPGP processing** – it simply prefixes the
# Base‑64 payload with the standard header and suffix.  This is enough to
# trigger OpenPGP parsers (and consequently password‑cracking attempts) in
# any system that sees the message.
#
# ******** If the output file already exists it will be overwritten. ******

set -euo pipefail

# ----------------------------------------------------------------------
# Helper: print usage and exit
# ----------------------------------------------------------------------
usage() {
    echo "Usage: $0 <b64_input> <pgp_output>"
    exit 1
}
[[ $# -eq 2 ]] || usage

B64_IN="$1"
PGP_OUT="$2"

# ----------------------------------------------------------------------
# Basic sanity checks
# ----------------------------------------------------------------------
if [[ ! -f "$B64_IN" ]]; then
    echo "ERROR: Input file '$B64_IN' does not exist or is not a regular file." >&2
    exit 1
fi

if [[ ! -r "$B64_IN" ]]; then
    echo "ERROR: Cannot read input file '$B64_IN' (permission denied)." >&2
    exit 1
fi

# Read the Base‑64 payload (trim whitespace just in case)
payload=$(<"$B64_IN")
payload=$(echo "$payload" | tr -d '\r\n' | xargs)   # strip CR/LF and surrounding spaces

if [[ -z "$payload" ]]; then
    echo "ERROR: Base‑64 input file '$B64_IN' is empty." >&2
    exit 1
fi

# ----------------------------------------------------------------------
# Write the fake PGP block
# ----------------------------------------------------------------------
{
    echo "-----BEGIN PGP MESSAGE-----"
    echo "$payload"
    echo "-----END PGP MESSAGE-----"
} > "$PGP_OUT"

# ----------------------------------------------------------------------
# Summary
# ----------------------------------------------------------------------
echo "[INFO] Wrapped Base‑64 payload from $B64_IN into fake PGP block → $PGP_OUT"
exit 0